<?php
require_once __DIR__ . '/_auth.php';
require_login();
require_once __DIR__ . '/_db.php';

header('Content-Type: application/json; charset=UTF-8');

function js_out($arr, $code=200){
  http_response_code($code);
  echo json_encode($arr, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);
  exit;
}

$meter_id = isset($_GET['meter_id']) ? (int)$_GET['meter_id'] : 0;
$start = trim((string)($_GET['start'] ?? ''));
$end   = trim((string)($_GET['end'] ?? ''));
$mode  = trim((string)($_GET['mode'] ?? 'power')); // power|energy|monthly|compare|alarms|peaks

if ($start === '') $start = date('Y-m-d 00:00:00');
if ($end === '')   $end   = date('Y-m-d 23:59:59');

// normalize (if only date)
if (preg_match('/^\d{4}-\d{2}-\d{2}$/',$start)) $start .= ' 00:00:00';
if (preg_match('/^\d{4}-\d{2}-\d{2}$/',$end))   $end   .= ' 23:59:59';

try {
  if ($mode === 'monthly') {
    // monthly consumption = max-min kwh_import_total per month
    $sql = "
      SELECT meter_id, DATE_FORMAT(ts, '%Y-%m') ym,
             MIN(kwh_import_total) kwh_start,
             MAX(kwh_import_total) kwh_end
      FROM meter_energy
      WHERE ts BETWEEN ? AND ?
      ".($meter_id>0?" AND meter_id=? ":"")."
      GROUP BY meter_id, DATE_FORMAT(ts, '%Y-%m')
      ORDER BY ym ASC
    ";
    $params = [$start, $end];
    if ($meter_id>0) $params[] = $meter_id;
    $st = $pdo->prepare($sql);
    $st->execute($params);
    $rows = $st->fetchAll(PDO::FETCH_ASSOC);

    $byMonth = [];
    foreach($rows as $r){
      $ym = $r['ym'];
      $mid = (int)$r['meter_id'];
      $cons = null;
      if ($r['kwh_start'] !== null && $r['kwh_end'] !== null) $cons = (float)$r['kwh_end'] - (float)$r['kwh_start'];
      if (!isset($byMonth[$ym])) $byMonth[$ym] = ['month'=>$ym,'total'=>0,'meters'=>[]];
      $byMonth[$ym]['meters'][$mid] = $cons;
      if ($cons !== null) $byMonth[$ym]['total'] += $cons;
    }
    js_out(['ok'=>true,'mode'=>'monthly','items'=>array_values($byMonth)]);
  }

  if ($mode === 'compare') {
    // per-meter comparison for selected range
    $params = [$start,$end,$start,$end];
    $where = "WHERE m.is_active=1";
    if ($meter_id>0){
      $where .= " AND m.id=?";
      $params[] = $meter_id;
    }
    $sql = "
      SELECT
        m.id AS meter_id,
        m.name,
        m.ip,
        m.port,
        e.kwh_total,
        h.kw_peak,
        h.pf_avg,
        h.online_pct
      FROM meters m
      LEFT JOIN (
        SELECT meter_id, (MAX(kwh_import_total)-MIN(kwh_import_total)) AS kwh_total
        FROM meter_energy
        WHERE ts BETWEEN ? AND ?
        GROUP BY meter_id
      ) e ON e.meter_id = m.id
      LEFT JOIN (
        SELECT meter_id,
               MAX(p_total_kw) AS kw_peak,
               AVG(pf) AS pf_avg,
               AVG(CASE WHEN online=1 THEN 1 ELSE 0 END) * 100 AS online_pct
        FROM meter_history
        WHERE ts BETWEEN ? AND ?
        GROUP BY meter_id
      ) h ON h.meter_id = m.id
      $where
      ORDER BY COALESCE(e.kwh_total,0) DESC, COALESCE(h.kw_peak,0) DESC
    ";
    $st = $pdo->prepare($sql);
    $st->execute($params);
    js_out(['ok'=>true,'mode'=>'compare','items'=>$st->fetchAll(PDO::FETCH_ASSOC)]);
  }

  if ($mode === 'energy') {
    // daily consumption = last - first kwh_import_total per day
    $sql = "
      SELECT meter_id, DATE(ts) d,
             MIN(kwh_import_total) kwh_start,
             MAX(kwh_import_total) kwh_end
      FROM meter_energy
      WHERE ts BETWEEN ? AND ?
      ".($meter_id>0?" AND meter_id=? ":"")."
      GROUP BY meter_id, DATE(ts)
      ORDER BY d ASC
    ";
    $params = [$start, $end];
    if ($meter_id>0) $params[] = $meter_id;
    $st = $pdo->prepare($sql);
    $st->execute($params);
    $rows = $st->fetchAll(PDO::FETCH_ASSOC);

    $byDay = [];
    foreach($rows as $r){
      $d = $r['d'];
      $mid = (int)$r['meter_id'];
      $cons = null;
      if ($r['kwh_start'] !== null && $r['kwh_end'] !== null) $cons = (float)$r['kwh_end'] - (float)$r['kwh_start'];
      if (!isset($byDay[$d])) $byDay[$d] = ['day'=>$d,'total'=>0,'meters'=>[]];
      $byDay[$d]['meters'][$mid] = $cons;
      if ($cons !== null) $byDay[$d]['total'] += $cons;
    }
    js_out(['ok'=>true,'mode'=>'energy','items'=>array_values($byDay)]);
  }

  if ($mode === 'peaks') {
    $sql = "
      SELECT meter_id,
             MAX(p_total_kw) kw_peak,
             MIN(pf) pf_min,
             MIN(vln1_v) v_min,
             MAX(vln1_v) v_max
      FROM meter_history
      WHERE ts BETWEEN ? AND ?
      ".($meter_id>0?" AND meter_id=? ":"")."
      GROUP BY meter_id
      ORDER BY kw_peak DESC
    ";
    $params = [$start,$end];
    if ($meter_id>0) $params[] = $meter_id;
    $st=$pdo->prepare($sql);
    $st->execute($params);
    js_out(['ok'=>true,'mode'=>'peaks','items'=>$st->fetchAll(PDO::FETCH_ASSOC)]);
  }

  if ($mode === 'alarms') {
    $st = $pdo->prepare("
      SELECT ts, level, source, meter_id, message
      FROM system_events
      WHERE ts BETWEEN ? AND ?
        AND level IN ('WARN','ERROR')
        ".($meter_id>0?" AND meter_id=? ":"")."
      ORDER BY ts DESC
      LIMIT 500
    ");
    $params = [$start,$end];
    if ($meter_id>0) $params[] = $meter_id;
    $st->execute($params);
    js_out(['ok'=>true,'mode'=>'alarms','items'=>$st->fetchAll(PDO::FETCH_ASSOC)]);
  }

  // power (timeseries)
  $st = $pdo->prepare("
    SELECT ts, online, p_total_kw, pf, vln1_v, vln2_v, vln3_v, il1_a, il2_a, il3_a
    FROM meter_history
    WHERE ts BETWEEN ? AND ?
      ".($meter_id>0?" AND meter_id=? ":"")."
    ORDER BY ts ASC
    LIMIT 6000
  ");
  $params = [$start,$end];
  if ($meter_id>0) $params[] = $meter_id;
  $st->execute($params);
  js_out(['ok'=>true,'mode'=>'power','items'=>$st->fetchAll(PDO::FETCH_ASSOC)]);
} catch (Throwable $e){
  js_out(['ok'=>false,'error'=>$e->getMessage()], 500);
}
