<?php
require_once __DIR__ . '/_auth.php';
require_login();
require_once __DIR__ . '/_db.php';

date_default_timezone_set('Europe/Istanbul');

@ini_set('output_buffering','off');
@ini_set('zlib.output_compression', 0);
@ini_set('implicit_flush', 1);
@set_time_limit(0);

header('Content-Type: text/event-stream; charset=UTF-8');
header('Cache-Control: no-cache, no-transform');
header('Connection: keep-alive');

function sse_send(string $event, array $data): void {
  echo "event: {$event}\n";
  echo 'data: ' . json_encode($data, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES) . "\n\n";
  @ob_flush();
  @flush();
}

$meter_id = isset($_GET['meter_id']) ? (int)$_GET['meter_id'] : 0;
if ($meter_id <= 0) {
  sse_send('tick', ['ok'=>false,'error'=>'meter_id required']);
  exit;
}

// offline threshold (sec)
$offline_after_sec = 5;
try {
  $stThr = $pdo->prepare("SELECT offline_after_sec FROM meter_thresholds WHERE meter_id=? LIMIT 1");
  $stThr->execute([$meter_id]);
  $thr = $stThr->fetch(PDO::FETCH_ASSOC);
  if ($thr && (int)$thr['offline_after_sec'] > 0) $offline_after_sec = (int)$thr['offline_after_sec'];
} catch (Throwable $e) {
  // ignore
}

$started = time();
$max_runtime = 3600; // 1 hour (browser reconnects if needed)

while (true) {
  if (connection_aborted()) break;
  if ((time() - $started) > $max_runtime) break;

  $t0 = microtime(true);
  try {
    $st = $pdo->prepare("\
      SELECT m.id AS meter_id, m.name, m.ip, m.port,
             l.ts, l.online, l.rtt_ms, l.p_total_kw, l.pf, l.kwh_import_total
      FROM meters m
      LEFT JOIN meter_last l ON l.meter_id=m.id
      WHERE m.id=?
      LIMIT 1
    ");
    $st->execute([$meter_id]);
    $r = $st->fetch(PDO::FETCH_ASSOC);

    $now = new DateTime('now');
    $ts = $r['ts'] ?? null;
    $age = null;
    if ($ts) {
      $dt = new DateTime($ts);
      $age = max(0, $now->getTimestamp() - $dt->getTimestamp());
    }

    $isOnline = false;
    if ($ts && $age !== null && $age <= $offline_after_sec) {
      $isOnline = ((int)($r['online'] ?? 0) === 1);
    }

    $payload = [
      'ok' => true,
      'meter_id' => (int)($r['meter_id'] ?? $meter_id),
      'name' => (string)($r['name'] ?? ''),
      'ip' => (string)($r['ip'] ?? ''),
      'port' => (int)($r['port'] ?? 502),
      'ts' => $ts,
      'age_sec' => $age,
      'offline_after_sec' => $offline_after_sec,
      'online' => $isOnline,
      'rtt_ms' => isset($r['rtt_ms']) ? (int)$r['rtt_ms'] : null,
      'p_total_kw' => isset($r['p_total_kw']) ? (float)$r['p_total_kw'] : null,
      'pf' => isset($r['pf']) ? (float)$r['pf'] : null,
      'kwh_import_total' => isset($r['kwh_import_total']) ? (float)$r['kwh_import_total'] : null,
    ];

    $payload['server_ts'] = (new DateTime('now'))->format('Y-m-d H:i:s');
    $payload['loop_ms'] = (int)round((microtime(true) - $t0) * 1000);

    sse_send('tick', $payload);
  } catch (Throwable $e) {
    sse_send('tick', ['ok'=>false,'error'=>$e->getMessage()]);
  }

  sleep(1);
}

sse_send('tick', ['ok'=>false,'error'=>'stream ended']);
